<?php
/**
 * Custom Script Manager
 *
 * @package     CSManager
 * @author      Marjan Mijic
 * @copyright   2025 Pannonic.com
 * @license     GPL-2.0-or-later
 *
 * @wordpress-plugin
 * Plugin Name: Custom Script Manager
 * Plugin URI:  https://pannonic.com/wordpress-script-manager/
 * Description: WordPress Script Manager/Editor for easy management and editing of custom HTML, JavaScript, CSS, and PHP code snippets. Allows PHP injection without a child theme. Features the Ace Editor with added version control and realtime CSS color preivew. Allows the user to specify where and when scripts run on the site by selecting specific pages, sections or element selectors. Includes advanced search functionality, allowing search through script content.
 * Version:     1.3
 * Author:      Marjan Mijic
 * Author URI:  https://pannonic.com/wordpress-script-manager/
 * Text Domain: csmanager
 * License:     GPL v2 or later
 * License URI: http://www.gnu.org/licenses/gpl-2.0.txt
 */

if (!defined('ABSPATH')) exit; // Exit if accessed directly

require_once __DIR__ . '/inc/plugin-update-checker.php';
use YahnisElsts\PluginUpdateChecker\v5\PucFactory;

class CustomScriptManager {
    private $script_positions = [
        'header_top'    => ['hook' => 'wp_head',        'priority' => 1],
        'header_middle' => ['hook' => 'wp_head',        'priority' => 1],
        'header_bottom' => ['hook' => 'wp_head',        'priority' => 1],
        'body_top'      => ['hook' => 'wp_body_open',   'priority' => 1],
        'body_bottom'   => ['hook' => 'wp_body_open',   'priority' => 1],
        'footer_top'    => ['hook' => 'wp_footer',      'priority' => 1],
        'footer_bottom' => ['hook' => 'wp_footer',      'priority' => 1],

        // Backend (Admin)
        'admin'         => ['hook' => 'admin_enqueue_scripts', 'priority' => 1],

        // Block Editor (Gutenberg)
        'blockeditor'   => ['hook' => 'enqueue_block_editor_assets', 'priority' => 1],

        // Login Page
        'login'         => ['hook' => 'login_enqueue_scripts', 'priority' => 1],

        // Add the new element position type
        'element'       => ['hook' => 'wp_footer', 'priority' => 99], // Use wp_footer to ensure DOM is loaded
    ];

    public function __construct() {
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), [$this, 'add_plugin_settings_link']);
        add_action('admin_enqueue_scripts', [$this, 'load_admin_assets']);

        // Add the update check
        add_action('admin_init', [$this, 'check_for_updates']);

        // Register hooks for all script positions
        foreach ($this->script_positions as $position => $details) {
            add_action($details['hook'], [$this, 'inject_scripts'], $details['priority']);
        }
        
        // Register additional hooks for admin, blockeditor, and login contexts
        add_action('admin_head', [$this, 'inject_scripts'], 1);
        add_action('admin_footer', [$this, 'inject_scripts'], 1);
        add_action('enqueue_block_editor_assets', [$this, 'inject_scripts'], 1);
        add_action('login_head', [$this, 'inject_scripts'], 1);
        add_action('login_footer', [$this, 'inject_scripts'], 1);

        // All AJAX actions
        add_action('wp_ajax_save_custom_script', [$this, 'save_script']);
        add_action('wp_ajax_get_custom_scripts', [$this, 'get_scripts']);
        add_action('wp_ajax_delete_custom_script', [$this, 'delete_script']);
        add_action('wp_ajax_toggle_custom_script', [$this, 'toggle_script']);
        add_action('wp_ajax_edit_custom_script', [$this, 'edit_script']);
        add_action('wp_ajax_update_custom_script', [$this, 'update_script']);
        add_action('wp_ajax_reorder_custom_script', [$this, 'reorder_custom_script']);
        add_action('wp_ajax_get_script_history', [$this, 'get_script_history']);
        add_action('wp_ajax_restore_script_version', [$this, 'restore_script_version']);
        add_action('wp_ajax_get_script_version', [$this, 'get_script_version']);
        add_action('wp_ajax_clear_script_history', [$this, 'clear_script_history']);
        
        // Register shortcode
        add_shortcode('csmanager', [$this, 'render_shortcode']);
    }

    public function add_admin_menu() {
        add_menu_page('Custom Scripts', 'CSManager', 'manage_options', 'custom-scripts', [$this, 'admin_page'], 'dashicons-editor-code');
    }

    public function load_admin_assets($hook) {

        $nonce = wp_create_nonce('custom-script-manager');

        if ($hook !== 'toplevel_page_custom-scripts') return;
        // Add version timestamp to prevent caching
        $version = time();
        wp_enqueue_script('ace-editor', 'https://cdnjs.cloudflare.com/ajax/libs/ace/1.4.12/ace.js', [], null, true);
        wp_enqueue_script('custom-script-manager', plugin_dir_url(__FILE__) . 'assets/js/admin-script.js', ['jquery'], $version, true);
        wp_localize_script('custom-script-manager', 'csm_ajax', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'positions' => array_keys($this->script_positions),
            'nonce' => $nonce,
            'cache_bust' => $version // Add cache-busting parameter
        ]);
        wp_enqueue_style('custom-script-style', plugin_dir_url(__FILE__) . 'assets/css/admin-style.css', [], $version);

    }

    public function admin_page() {
        include __DIR__.'/assets/admin_page.php';
    }

    public function save_script() {
        if (!current_user_can('manage_options')) wp_die('Unauthorized');
        
        // Verify nonce for CSRF protection
        if (!check_ajax_referer('custom-script-manager', 'nonce', false)) {
            wp_send_json_error(['message' => 'Invalid security token']);
            return;
        }
        
        $phpDate = new DateTime();
        $mysqlTimestamp = $phpDate->format('Y-m-d H:i:s');
        $scripts = get_option('custom_scripts', []);
        
        $script_id = uniqid('script_'); // Generate unique ID for new script
        
        // Handle context as array
        $context = [];
        if (isset($_POST['context']) && is_array($_POST['context'])) {
            $context = array_map('sanitize_text_field', $_POST['context']);
        } elseif (isset($_POST['context'])) {
            $context = [sanitize_text_field($_POST['context'])];
        }
        
        // Validate and decode base64 code
        if (!isset($_POST['code']) || empty($_POST['code'])) {
            wp_send_json_error(['message' => 'Script code is required']);
            return;
        }
        $decoded_code = base64_decode($_POST['code'], true);
        if ($decoded_code === false) {
            wp_send_json_error(['message' => 'Invalid code format']);
            return;
        }
        
        $new_script = [
            'id' => $script_id,
            'name' => sanitize_text_field($_POST['name'] ?? 'Unnamed'),
            'type' => sanitize_text_field($_POST['type'] ?? 'javascript'),
            'code' => $decoded_code,
            'position' => sanitize_text_field($_POST['position']),
            'location' => sanitize_text_field($_POST['location']),
            'pages' => sanitize_text_field($_POST['pages'] ?? ''),
            'shortcode_id' => sanitize_text_field($_POST['shortcode_id'] ?? ''),
            'selector' => sanitize_text_field($_POST['selector'] ?? ''),
            'context' => $context,
            'css_minimize' => isset($_POST['css_minimize']) && ($_POST['css_minimize'] === 'true' || $_POST['css_minimize'] === true),
            'js_base64' => isset($_POST['js_base64']) && ($_POST['js_base64'] === 'true' || $_POST['js_base64'] === true),
            'created' => $mysqlTimestamp,
            'active' => true
        ];
        
        $scripts[] = $new_script;
        update_option('custom_scripts', $scripts);

        // Save initial version to history
        $histories = get_option('custom_scripts_history', []);
        if (!isset($histories[$script_id])) {
            $histories[$script_id] = [];
        }
        $histories[$script_id][] = [
            'timestamp' => $mysqlTimestamp,
            'code' => $new_script['code'],
            'name' => $new_script['name'],
            'type' => $new_script['type'],
            'position' => $new_script['position'],
            'location' => $new_script['location'],
            'pages' => $new_script['pages'],
            'selector' => $new_script['selector'],
            'context' => $new_script['context'],
            'shortcode_id' => $new_script['shortcode_id'],
            'css_minimize' => $new_script['css_minimize'],
            'js_base64' => $new_script['js_base64']
        ];
        update_option('custom_scripts_history', $histories);
        
        wp_send_json_success(['message' => 'Script saved']);
    }

    public function get_scripts() {
        if (!current_user_can('manage_options')) wp_die('Unauthorized');
        
        // Verify nonce for CSRF protection
        if (!check_ajax_referer('custom-script-manager', 'nonce', false)) {
            wp_send_json_error(['message' => 'Invalid security token']);
            return;
        }
        
        $scripts = get_option('custom_scripts', []);
        wp_send_json_success($scripts);
    }

    public function delete_script() {
        if (!current_user_can('manage_options')) wp_die('Unauthorized');
        
        // Verify nonce for CSRF protection
        if (!check_ajax_referer('custom-script-manager', 'nonce', false)) {
            wp_send_json_error(['message' => 'Invalid security token']);
            return;
        }
        
        $scripts = get_option('custom_scripts', []);
        $index = intval($_POST['index'] ?? -1);
        if (isset($scripts[$index])) {
            unset($scripts[$index]);
            $scripts = array_values($scripts);
            update_option('custom_scripts', $scripts);
            wp_send_json_success(['message' => 'Script deleted']);
        }
        wp_send_json_error(['message' => 'Invalid script index']);
    }

    public function toggle_script() {
        if (!current_user_can('manage_options')) wp_die('Unauthorized');
        
        // Verify nonce for CSRF protection
        if (!check_ajax_referer('custom-script-manager', 'nonce', false)) {
            wp_send_json_error(['message' => 'Invalid security token']);
            return;
        }
        
        $scripts = get_option('custom_scripts', []);
        $index = intval($_POST['index'] ?? -1);
        if (isset($scripts[$index])) {
            $scripts[$index]['active'] = !$scripts[$index]['active'];
            update_option('custom_scripts', $scripts);
            wp_send_json_success(['message' => 'Script status updated']);
        }
        wp_send_json_error(['message' => 'Invalid script index']);
    }

    public function edit_script() {
        if (!current_user_can('manage_options')) wp_die('Unauthorized');
        
        // Verify nonce for CSRF protection
        if (!check_ajax_referer('custom-script-manager', 'nonce', false)) {
            wp_send_json_error(['message' => 'Invalid security token']);
            return;
        }
        
        $scripts = get_option('custom_scripts', []);
        $index = intval($_POST['index'] ?? -1);
        if (isset($scripts[$index])) {
            wp_send_json_success($scripts[$index]);
        }
        wp_send_json_error(['message' => 'Invalid script index']);
    }

    public function update_script() {
        if (!current_user_can('manage_options')) wp_die('Unauthorized');
        
        // Verify nonce for CSRF protection
        if (!check_ajax_referer('custom-script-manager', 'nonce', false)) {
            wp_send_json_error(['message' => 'Invalid security token']);
            return;
        }
        
        $scripts = get_option('custom_scripts', []);
        $index = intval($_POST['index'] ?? -1);
        
        if (isset($scripts[$index])) {
            $phpDate = new DateTime();
            $mysqlTimestamp = $phpDate->format('Y-m-d H:i:s');
            
            // Get script ID (or create one if it doesn't exist)
            $script_id = isset($scripts[$index]['id']) ? $scripts[$index]['id'] : uniqid('script_');
            
            // Handle context as array
            $context = [];
            if (isset($_POST['context']) && is_array($_POST['context'])) {
                $context = array_map('sanitize_text_field', $_POST['context']);
            } elseif (isset($_POST['context'])) {
                $context = [sanitize_text_field($_POST['context'])];
            }
            
            // Validate and decode base64 code
            if (!isset($_POST['code']) || empty($_POST['code'])) {
                wp_send_json_error(['message' => 'Script code is required']);
                return;
            }
            $decoded_code = base64_decode($_POST['code'], true);
            if ($decoded_code === false) {
                wp_send_json_error(['message' => 'Invalid code format']);
                return;
            }
            
            $updated_script = [
                'id' => $script_id,
                'name' => sanitize_text_field($_POST['name'] ?? 'Unnamed'),
                'type' => sanitize_text_field($_POST['type'] ?? 'javascript'),
                'code' => $decoded_code,
                'position' => sanitize_text_field($_POST['position']),
                'location' => sanitize_text_field($_POST['location']),
                'pages' => sanitize_text_field($_POST['pages'] ?? ''),
                'shortcode_id' => sanitize_text_field($_POST['shortcode_id'] ?? ''),
                'selector' => sanitize_text_field($_POST['selector'] ?? ''),
                'context' => $context,
                'css_minimize' => isset($_POST['css_minimize']) && ($_POST['css_minimize'] === 'true' || $_POST['css_minimize'] === true),
                'js_base64' => isset($_POST['js_base64']) && ($_POST['js_base64'] === 'true' || $_POST['js_base64'] === true),
                'created' => sanitize_text_field($_POST['created']),
                'modified' => $mysqlTimestamp,
                'active' => $scripts[$index]['active']
            ];
            
            $scripts[$index] = $updated_script;
            update_option('custom_scripts', $scripts);

            // Save version to history
            $histories = get_option('custom_scripts_history', []);
            if (!isset($histories[$script_id])) {
                $histories[$script_id] = [];
            }
            $histories[$script_id][] = [
                'timestamp' => $mysqlTimestamp,
                'code' => $updated_script['code'],
                'name' => $updated_script['name'],
                'type' => $updated_script['type'],
                'position' => $updated_script['position'],
                'location' => $updated_script['location'],
                'pages' => $updated_script['pages'],
                'selector' => $updated_script['selector'],
                'context' => $updated_script['context'],
                'shortcode_id' => $updated_script['shortcode_id'],
                'css_minimize' => $updated_script['css_minimize'],
                'js_base64' => $updated_script['js_base64']
            ];
            update_option('custom_scripts_history', $histories);
            
            wp_send_json_success(['message' => 'Script updated']);
        }
        wp_send_json_error(['message' => 'Invalid script index']);
    }

    public function reorder_custom_script() {
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }

        // Verify nonce if provided
        if (!check_ajax_referer('custom-script-manager', 'nonce', false)) {
            wp_send_json_error(['message' => 'Invalid nonce']);
            return;
        }

        $scripts = get_option('custom_scripts', []);
        $index = intval($_POST['index']);
        $direction = sanitize_text_field($_POST['direction']);

        // Validate index
        if (!isset($scripts[$index])) {
            wp_send_json_error(['message' => 'Invalid script index']);
            return;
        }

        // Can't move first item up or last item down
        if (($index === 0 && $direction === 'up') || 
            ($index === count($scripts) - 1 && $direction === 'down')) {
            wp_send_json_error(['message' => 'Cannot move item further']);
            return;
        }

        // Perform the swap
        if ($direction === 'up' && $index > 0) {
            $temp = $scripts[$index - 1];
            $scripts[$index - 1] = $scripts[$index];
            $scripts[$index] = $temp;
        } else if ($direction === 'down' && $index < count($scripts) - 1) {
            $temp = $scripts[$index + 1];
            $scripts[$index + 1] = $scripts[$index];
            $scripts[$index] = $temp;
        }

        update_option('custom_scripts', $scripts);
        wp_send_json_success(['message' => 'Script reordered successfully']);
    }

    public function inject_scripts() {

        static $already_run = [];  // Track which hooks have been processed

        $current_hook = current_action();
        $current_priority = has_filter($current_hook, [$this, 'inject_scripts']);

        // Create a unique key for this hook+priority combination
        $hook_key = $current_hook . '_' . $current_priority;
        // If we've already processed this hook+priority, skip it
        /*if (isset($already_run[$hook_key])) {
            return;
        }*/
        $already_run[$hook_key] = true;  // Mark this hook as processed

        $scripts = get_option('custom_scripts', []);

        foreach ($scripts as $script) {

            // Add default values to prevent undefined array key warnings
            $script = array_merge([
                'active' => false,
                'position'  => '',
                'location'  => 'all',
                'pages'     => '',
                'type'      => '',
                'code'      => '',
                'context'   => [],
                'selector'  => '' // Add selector field
            ], $script);

            if (!$script['active']) continue;

            // Handle context as array - check if current context matches
            $context = is_array($script['context']) ? $script['context'] : (!empty($script['context']) ? [$script['context']] : []);
            
            // Backward compatibility: if position is admin/blockeditor/login (legacy), use that as context
            if (empty($context) && in_array($script['position'], ['admin', 'blockeditor', 'login'])) {
                $context = [$script['position']];
            }
            
            // If no context is selected, skip this script (don't run it)
            if (empty($context)) {
                continue;
            }
            
            // Check if we're in the right context
            $current_context = [];
            
            // Determine current context
            if (is_admin()) {
                // Check if we're in block editor
                if (function_exists('get_current_screen')) {
                    $screen = get_current_screen();
                    if ($screen && $screen->is_block_editor()) {
                        $current_context[] = 'blockeditor';
                    } else {
                        $current_context[] = 'admin';
                    }
                } else {
                    $current_context[] = 'admin';
                }
            } else {
                $current_context[] = 'frontend';
            }
            
            // Check if we're on login page
            if (in_array($GLOBALS['pagenow'], ['wp-login.php', 'wp-register.php'])) {
                $current_context[] = 'login';
            }
            
            // Check if any current context matches script context
            $context_match = false;
            foreach ($current_context as $current) {
                if (in_array($current, $context)) {
                    $context_match = true;
                    break;
                }
            }
            
            // If no context match, skip this script
            if (!$context_match) {
                continue;
            }

            // Validate position exists before accessing
            if (empty($script['position']) || !isset($this->script_positions[$script['position']])) {
                continue;
            }

            $position_details = $this->script_positions[$script['position']];
            
            // Map context + position to hooks
            $context_hook_map = [
                'admin' => [
                    'header_top' => 'admin_head',
                    'body_top' => 'admin_head', // Admin doesn't have body_open, use head
                    'footer_top' => 'admin_footer',
                    'element' => 'admin_footer', // Elements use footer for DOM ready
                    'admin' => 'admin_enqueue_scripts' // Legacy position
                ],
                'blockeditor' => [
                    'header_top' => 'enqueue_block_editor_assets',
                    'body_top' => 'enqueue_block_editor_assets',
                    'footer_top' => 'enqueue_block_editor_assets',
                    'element' => 'enqueue_block_editor_assets',
                    'blockeditor' => 'enqueue_block_editor_assets' // Legacy position
                ],
                'login' => [
                    'header_top' => 'login_head',
                    'body_top' => 'login_head', // Login doesn't have body_open, use head
                    'footer_top' => 'login_footer',
                    'element' => 'login_footer', // Elements use footer for DOM ready
                    'login' => 'login_enqueue_scripts' // Legacy position
                ],
                'frontend' => [
                    'header_top' => 'wp_head',
                    'body_top' => 'wp_body_open',
                    'footer_top' => 'wp_footer',
                    'element' => 'wp_footer' // Elements use footer for DOM ready
                ]
            ];
            
            // Determine which hook to use based on context and position
            $expected_hook = null;
            if (!empty($context)) {
                // Find the first matching context that has a hook mapping for this position
                foreach ($context as $ctx) {
                    if (isset($context_hook_map[$ctx][$script['position']])) {
                        $expected_hook = $context_hook_map[$ctx][$script['position']];
                        break;
                    }
                }
            }
            
            // If no context-specific hook found, use default position hook
            if ($expected_hook === null) {
                $expected_hook = $position_details['hook'];
            }
            
            // Check if current hook matches expected hook
            if ($expected_hook !== $current_hook) {
                continue;
            }

            // Skip scripts with shortcode location (handled by shortcode handler)
            if ($script['location'] === 'shortcode') {
                continue;
            }
            
            // Check page targeting with proper validation
            if ($script['location'] === 'all' || 
                ($script['location'] === 'specific' && !empty($script['pages']) && is_page(explode(',', $script['pages']))) ||
                ($script['location'] === 'range' && !empty($script['pages']) && $this->is_in_page_range($script['pages']))) {

                if ($script['position'] === 'element' && !empty($script['selector'])) {
                    // Handle element insertion
                    $this->inject_into_element($script);
                } else {
                    // Existing injection logic
                    if ($script['type'] === 'javascript') {
                        $cleaned_code = $script['code'];
                        // Base64 encode if option is enabled
                        if (!empty($script['js_base64'])) {
                            $cleaned_code = base64_encode($cleaned_code);
                            echo "<script src=\"data:text/javascript;base64," . esc_attr($cleaned_code) . "\"></script>";
                        } else {
                            echo "<script>";
                            echo $cleaned_code;
                            echo "</script>";
                        }
                    }
                    if ($script['type'] === 'html') echo stripslashes($script['code']);
                    if ($script['type'] === 'css') {
                        $css_code = stripslashes($script['code']);
                        // Minimize CSS if option is enabled
                        if (!empty($script['css_minimize'])) {
                            $css_code = $this->minify_css($css_code);
                        }
                        echo "<style>" . $css_code . "</style>";
                    }
                    if ($script['type'] === 'php') {
                        // SECURITY WARNING: eval() is used here to allow PHP code execution.
                        // This is an intentional feature but poses security risks if untrusted users
                        // have access to this plugin. Only users with 'manage_options' capability
                        // can create/edit scripts, which should be limited to trusted administrators.
                        // Modify the head content using wp_head filter
                        if ($position_details['hook'] === 'wp_head') {
                            add_action('wp_head', function() use ($script) {
                                eval("?>{$script['code']}");
                            });
                        } else {
                            eval("?>{$script['code']}");
                        }
                    }
                }
            }
        }
    }

    private function inject_into_element($script) {
        // Create a script that will inject the content into the selected elements
        $selector = esc_js($script['selector']);
        
        // Prepare content based on type and options
        $content = $script['code'];
        if ($script['type'] === 'css' && !empty($script['css_minimize'])) {
            $content = $this->minify_css($content);
        }
        if ($script['type'] === 'javascript' && !empty($script['js_base64'])) {
            $content = base64_encode($content);
        }
        
        $content_escaped = esc_js($content);
        
        echo "<script>
        document.addEventListener('DOMContentLoaded', function() {
            const elements = document.querySelectorAll('" . $selector . "');
            elements.forEach(function(element) {
                if (element) {";

        // Handle different content types
        switch ($script['type']) {
            case 'html':
                echo "element.innerHTML += `" . $content_escaped . "`;";
                break;
            
            case 'javascript':
                if (!empty($script['js_base64'])) {
                    echo "
                    const scriptTag = document.createElement('script');
                    scriptTag.src = 'data:text/javascript;base64," . $content_escaped . "';
                    element.appendChild(scriptTag);";
                } else {
                    echo "
                    const scriptTag = document.createElement('script');
                    scriptTag.text = `" . $content_escaped . "`;
                    element.appendChild(scriptTag);";
                }
                break;
            
            case 'css':
                echo "
                const styleTag = document.createElement('style');
                styleTag.textContent = `" . $content_escaped . "`;
                element.appendChild(styleTag);";
                break;
            
            case 'php':
                // PHP can't be injected into elements client-side
                error_log('PHP injection into elements is not supported');
                break;
        }

        echo "
                }
            });
        });
        </script>";
    }

    private function is_in_page_range($range) {
        if (!is_page()) return false;
        $current_page_id = get_the_ID();
        list($start, $end) = array_map('intval', explode('-', $range));
        return $current_page_id >= $start && $current_page_id <= $end;
    }

    public function get_script_history() {
        if (!current_user_can('manage_options')) wp_die('Unauthorized');
        
        // Verify nonce for CSRF protection
        if (!check_ajax_referer('custom-script-manager', 'nonce', false)) {
            wp_send_json_error(['message' => 'Invalid security token']);
            return;
        }
        
        $script_id = sanitize_text_field($_POST['script_id'] ?? '');
        if (empty($script_id)) {
            wp_send_json_error(['message' => 'Script ID is required']);
            return;
        }
        
        $histories = get_option('custom_scripts_history', []);
        
        if (isset($histories[$script_id])) {
            wp_send_json_success($histories[$script_id]);
        }
        wp_send_json_error(['message' => 'No history found']);
    }

    public function restore_script_version() {
        if (!current_user_can('manage_options')) wp_die('Unauthorized');
        
        // Verify nonce for CSRF protection
        if (!check_ajax_referer('custom-script-manager', 'nonce', false)) {
            wp_send_json_error(['message' => 'Invalid security token']);
            return;
        }
        
        $script_id = sanitize_text_field($_POST['script_id'] ?? '');
        $version_timestamp = sanitize_text_field($_POST['version'] ?? '');
        
        if (empty($script_id) || empty($version_timestamp)) {
            wp_send_json_error(['message' => 'Script ID and version are required']);
            return;
        }
        
        $histories = get_option('custom_scripts_history', []);
        $scripts = get_option('custom_scripts', []);
        
        // Find script index by ID
        $script_index = array_search($script_id, array_column($scripts, 'id'));
        
        if ($script_index !== false && isset($histories[$script_id])) {
            // Find the version in history
            $version = array_filter($histories[$script_id], function($v) use ($version_timestamp) {
                return $v['timestamp'] === $version_timestamp;
            });
            
            if (!empty($version)) {
                $version = reset($version);
                $phpDate = new DateTime();
                $mysqlTimestamp = $phpDate->format('Y-m-d H:i:s');
                
                // Update script with historical version
                $scripts[$script_index] = array_merge($scripts[$script_index], [
                    'code' => $version['code'],
                    'name' => $version['name'],
                    'type' => $version['type'],
                    'position' => $version['position'],
                    'location' => $version['location'],
                    'pages' => $version['pages'],
                    'selector' => $version['selector'],
                    'context' => isset($version['context']) ? $version['context'] : [],
                    'shortcode_id' => isset($version['shortcode_id']) ? $version['shortcode_id'] : '',
                    'css_minimize' => isset($version['css_minimize']) ? $version['css_minimize'] : false,
                    'js_base64' => isset($version['js_base64']) ? $version['js_base64'] : false,
                    'modified' => $mysqlTimestamp
                ]);
                
                update_option('custom_scripts', $scripts);
                wp_send_json_success(['message' => 'Version restored']);
            }
        }
        
        wp_send_json_error(['message' => 'Version not found']);
    }

    public function get_script_version() {
        if (!current_user_can('manage_options')) wp_die('Unauthorized');
        
        // Verify nonce for CSRF protection
        if (!check_ajax_referer('custom-script-manager', 'nonce', false)) {
            wp_send_json_error(['message' => 'Invalid security token']);
            return;
        }
        
        $script_id = sanitize_text_field($_POST['script_id'] ?? '');
        $version_timestamp = sanitize_text_field($_POST['version'] ?? '');
        
        if (empty($script_id) || empty($version_timestamp)) {
            wp_send_json_error(['message' => 'Script ID and version are required']);
            return;
        }
        
        $histories = get_option('custom_scripts_history', []);
        
        if (isset($histories[$script_id])) {
            // Find the specific version
            $version = array_filter($histories[$script_id], function($v) use ($version_timestamp) {
                return $v['timestamp'] === $version_timestamp;
            });
            
            if (!empty($version)) {
                wp_send_json_success(reset($version));
                return;
            }
        }
        
        wp_send_json_error(['message' => 'Version not found']);
    }

    public function clear_script_history() {
        if (!current_user_can('manage_options')) wp_die('Unauthorized');
        
        // Verify nonce for CSRF protection
        if (!check_ajax_referer('custom-script-manager', 'nonce', false)) {
            wp_send_json_error(['message' => 'Invalid security token']);
            return;
        }
        
        $script_id = sanitize_text_field($_POST['script_id'] ?? '');
        if (empty($script_id)) {
            wp_send_json_error(['message' => 'Script ID is required']);
            return;
        }
        
        $histories = get_option('custom_scripts_history', []);
        
        if (isset($histories[$script_id])) {
            // Keep only the most recent version
            $latest = end($histories[$script_id]);
            $histories[$script_id] = [$latest];
            
            update_option('custom_scripts_history', $histories);
            wp_send_json_success(['message' => 'History cleared']);
        }
        
        wp_send_json_error(['message' => 'No history found']);
    }

    public function check_for_updates(){
        $myUpdateChecker = PucFactory::buildUpdateChecker(
            'https://pannonic.com/csmanager/details.json',
            __FILE__,
            'custom-script-manager'
        );
    }

    public function add_plugin_settings_link($links) {
        $settings_url = admin_url('admin.php?page=custom-scripts'); // Replace with your actual slug
        $settings_link = '<a href="' . esc_url($settings_url) . '">Settings</a>';
        array_unshift($links, $settings_link);
        return $links;
    }
}

new CustomScriptManager();